/*!
    \file    dci_ov2640.c
    \brief   DCI config file

    \version 2025-08-11, V2.9.0, demo for GD32F20x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "dci_ov2640.h"
#include "dci_ov2640_init_table.h"
#include "gd32f20x_dci.h"
#include "systick.h"

extern uint8_t capture_image[320 * 240 * 2];

/*!
    \brief      configure the DCI to interface with the camera module
    \param[in]  none
    \param[out] none
    \retval     none
*/
void dci_config(void)
{
    dci_parameter_struct dci_struct;
    dma_parameter_struct dma_struct;

    /* enable periphrals clock */
    rcu_periph_clock_enable(RCU_GPIOA);
    rcu_periph_clock_enable(RCU_GPIOB);
    rcu_periph_clock_enable(RCU_GPIOC);
    rcu_periph_clock_enable(RCU_GPIOD);
    rcu_periph_clock_enable(RCU_GPIOG);
    rcu_periph_clock_enable(RCU_GPIOH);
    rcu_periph_clock_enable(RCU_DCI);
    rcu_periph_clock_enable(RCU_AF);

    /* DCI GPIO remap configuration */
    gpio_pin_remap1_config(GPIO_PCF2, GPIO_PCF2_DCI_D5_PD3_REMAP | GPIO_PCF2_DCI_D0_PC6_REMAP |
                           GPIO_PCF2_DCI_D1_PC7_REMAP | GPIO_PCF2_DCI_VSYNC_PG9_REMAP |
                           GPIO_PCF2_DCI_HSYNC_PH8_REMAP, ENABLE);

    /* configure DCI_PIXCLK(PA6), DCI_VSYNC(PG9), DCI_HSYNC(PH8), DCI_D0(PC6), DCI_D1(PC7)
                 DCI_D2(PC8), DCI_D3(PC9), DCI_D4(PC11), DCI_D5(PD3), DCI_D6(PB8), DCI_D7(PB9) */
    gpio_init(GPIOA, GPIO_MODE_IN_FLOATING, GPIO_OSPEED_10MHZ, GPIO_PIN_6);

    gpio_init(GPIOB, GPIO_MODE_IN_FLOATING, GPIO_OSPEED_10MHZ, GPIO_PIN_8 | GPIO_PIN_9);

    gpio_init(GPIOC, GPIO_MODE_IN_FLOATING, GPIO_OSPEED_10MHZ, GPIO_PIN_6 | GPIO_PIN_7 | GPIO_PIN_8 |
              GPIO_PIN_9 | GPIO_PIN_11);

    gpio_init(GPIOD, GPIO_MODE_IN_FLOATING, GPIO_OSPEED_10MHZ, GPIO_PIN_3);

    gpio_init(GPIOG, GPIO_MODE_IN_FLOATING, GPIO_OSPEED_10MHZ, GPIO_PIN_9);

    gpio_init(GPIOH, GPIO_MODE_IN_FLOATING, GPIO_OSPEED_10MHZ, GPIO_PIN_8);

    /* DCI configuration */
    dci_struct.capture_mode = DCI_CAPTURE_MODE_CONTINUOUS;
    dci_struct.clock_polarity = DCI_CK_POLARITY_RISING;
    dci_struct.hsync_polarity = DCI_HSYNC_POLARITY_LOW;
    dci_struct.vsync_polarity = DCI_VSYNC_POLARITY_LOW;
    dci_struct.frame_rate = DCI_FRAME_RATE_ALL;
    dci_struct.interface_format = DCI_INTERFACE_FORMAT_8BITS;
    dci_init(&dci_struct);

    /* DCI DMA configuration */
    rcu_periph_clock_enable(RCU_DMA1);
    dma_deinit(DMA1, DMA_CH5);
    dma_struct.periph_addr = (uint32_t)DCI_DR_ADDRESS;
    dma_struct.memory_addr = (uint32_t)capture_image;
    dma_struct.direction = DMA_PERIPHERAL_TO_MEMORY;
    dma_struct.number = 38400;
    dma_struct.periph_inc = DMA_PERIPH_INCREASE_DISABLE;
    dma_struct.memory_inc = DMA_MEMORY_INCREASE_ENABLE;
    dma_struct.periph_width = DMA_PERIPHERAL_WIDTH_32BIT;
    dma_struct.memory_width = DMA_MEMORY_WIDTH_32BIT;
    dma_struct.priority = DMA_PRIORITY_HIGH;
    dma_init(DMA1, DMA_CH5, &dma_struct);
    dma_circulation_enable(DMA1, DMA_CH5);
}

/*!
    \brief      set DCI camera outsize
    \param[in]  width: outsize width
    \param[in]  height: outsize height
    \param[out] none
    \retval     0x00 or 0xFF
*/
uint8_t ov2640_outsize_set(uint16_t width, uint16_t height)
{
    uint16_t outh;
    uint16_t outw;
    uint8_t temp;
    /* make sure the width and height are integral multiples of 4 */
    if(width % 4) {
        return 0xFF;
    }
    if(height % 4) {
        return 0xFF;
    }
    outw = width / 4;
    outh = height / 4;
    /* configure camera register */
    dci_byte_write(0xFF, 0x00);
    dci_byte_write(0xE0, 0x04);
    dci_byte_write(0x5A, outw & 0xFF);
    dci_byte_write(0x5B, outh & 0xFF);
    temp = (outw >> 8) & 0x03;
    temp |= (outh >> 6) & 0x04;
    dci_byte_write(0x5C, temp);
    dci_byte_write(0xE0, 0x00);
    return 0;
}

/*!
    \brief      DCI camera initialization
    \param[in]  none
    \param[out] none
    \retval     0x00 or 0xFF
*/
uint8_t dci_ov2640_init(void)
{
    uint8_t i;
    sccb_config();
    dci_config();

    ckout0_init();
    delay_1ms(100);
    /* OV2640 reset */
    if(dci_byte_write(0xFF, 0x01) != 0) {
        return 0xFF;
    }
    if(dci_byte_write(0x12, 0x80) != 0) {
        return 0xFF;
    }
    delay_1ms(10);

    /* configure camera via ov2640_svga_init_reg_tbl */
    for(i = 0; i < sizeof(ov2640_svga_init_reg_tbl) / 2; i++) {
        if(0 != dci_byte_write(ov2640_svga_init_reg_tbl[i][0], ov2640_svga_init_reg_tbl[i][1])) {
            return 0xFF;
        }
    }

    delay_1ms(100);
    for(i = 0; i < (sizeof(ov2640_rgb565_reg_tbl) / 2); i++) {
        if(0 != dci_byte_write(ov2640_rgb565_reg_tbl[i][0], ov2640_rgb565_reg_tbl[i][1])) {
            return 0xFF;
        }
    }
    delay_1ms(100);
    ov2640_outsize_set(320, 240);
    return 0;
}

/*!
    \brief      ckout0 initialization
    \param[in]  none
    \param[out] none
    \retval     none
*/
void ckout0_init(void)
{
    /* enable periphrals clock and configure GPIO */
    rcu_periph_clock_enable(RCU_GPIOA);
    rcu_periph_clock_enable(RCU_AF);
    gpio_init(GPIOA, GPIO_MODE_AF_PP, GPIO_OSPEED_50MHZ, GPIO_PIN_8);

    /* configure CKOUT0 */
    rcu_ckout0_config(RCU_CKOUT0SRC_HXTAL, RCU_CKOUT0_DIV3);
}

/*!
    \brief      read the ov2640 manufacturer identifier
    \param[in]  ov2640id: pointer to the ov2640 manufacturer struct
    \param[out] none
    \retval     0x00 or 0xFF
*/
uint8_t dci_ov2640_id_read(ov2640_id_struct *ov2640id)
{
    uint8_t temp;
    dci_byte_write(0xFF, 0x01);
    if(0 != dci_byte_read(OV2640_MIDH, &temp)) {
        return 0xFF;
    }
    ov2640id->manufacturer_id1 = temp;
    if(0 != dci_byte_read(OV2640_MIDL, &temp)) {
        return 0xFF;
    }
    ov2640id->manufacturer_id2 = temp;
    if(0 != dci_byte_read(OV2640_VER, &temp)) {
        return 0xFF;
    }
    ov2640id->version = temp;
    if(0 != dci_byte_read(OV2640_PID, &temp)) {
        return 0xFF;
    }
    ov2640id->pid = temp;

    return 0x00;
}
