/*!
    \file    main.c
    \brief   I2S master send and slave receive communication using DMA

    \version 2025-08-08, V2.6.0, firmware for GD32F10x
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32f10x.h"
#include "gd32f10x_eval.h"

#define ARRAYSIZE          10

uint16_t i2s1_send_array[ARRAYSIZE] = {0xAAB1, 0xAAB2, 0xAAB3, 0xAAB4, 0xAAB5, 0xAAB6, 0xAAB7, 0xAAB8, 0xAAB9, 0xAABA};
uint16_t i2s2_receive_array[ARRAYSIZE];
uint32_t send_n = 0, receive_n = 0;
ErrStatus hxtal_stabilization;

void rcu_config(void);
void gpio_config(void);
void dma_config(void);
void spi_config(void);
ErrStatus memory_compare(uint16_t *src, uint16_t *dst, uint8_t length);

/*!
    \brief      main function
    \param[in]  none
    \param[out] none
    \retval     none
*/
int main(void)
{
    /* initialize the LED */
    gd_eval_led_init(LED2);

    /* enable peripheral clock */
    rcu_config();
    /* configure GPIO */
    gpio_config();
    /* configure DMA */
    dma_config();
    /* configure SPI */
    spi_config();

    /* enable SPI DMA */
    spi_dma_enable(SPI2, SPI_DMA_RECEIVE);
    spi_dma_enable(SPI1, SPI_DMA_TRANSMIT);

    /* enable DMA channel */
    /* I2S1_Rx DMA channel */
    dma_channel_enable(DMA1, DMA_CH0);
    /* I2S1_Tx DMA channel */
    dma_channel_enable(DMA0, DMA_CH4);

    /* enable SPI */
    i2s_enable(SPI2);
    i2s_enable(SPI1);

    /* wait DMA transmit completed */
    while(!dma_flag_get(DMA0, DMA_CH4, DMA_FLAG_FTF)) {
    }
    while(!dma_flag_get(DMA1, DMA_CH0, DMA_FLAG_FTF)) {
    }

    /* compare receive data with send data */
    if(ERROR != memory_compare(i2s2_receive_array, i2s1_send_array, ARRAYSIZE)) {
        gd_eval_led_on(LED2);
    } else {
        gd_eval_led_off(LED2);
    }

    while(1);
}

/*!
    \brief      configure different peripheral clocks
    \param[in]  none
    \param[out] none
    \retval     none
*/
void rcu_config(void)
{
    /* deinitialize the RCU */
    rcu_deinit();

    /* turn on the oscillator HXTAL */
    rcu_osci_on(RCU_HXTAL);
    hxtal_stabilization = rcu_osci_stab_wait(RCU_HXTAL);
    if(SUCCESS == hxtal_stabilization) {
        /* AHB = SYSCLK */
        rcu_ahb_clock_config(RCU_AHB_CKSYS_DIV1);
        /* APB1 = AHB/2 */
        rcu_apb1_clock_config(RCU_APB1_CKAHB_DIV2);
        /* APB2 = AHB/1 */
        rcu_apb2_clock_config(RCU_APB2_CKAHB_DIV1);
        /* CK_PLL1 = 25M */
        rcu_predv1_config(RCU_PREDV1_DIV8);
        rcu_pll1_config(RCU_PLL1_MUL8);
        /* turn on the oscillator PLL1 */
        rcu_osci_on(RCU_PLL1_CK);
        while(ERROR == rcu_osci_stab_wait(RCU_PLL1_CK));
        /* CK_PLL = 25M *3 = 75M*/
        rcu_predv0_config(RCU_PREDV0SRC_CKPLL1, RCU_PREDV0_DIV1);
        rcu_pll_config(RCU_PLLSRC_HXTAL, RCU_PLL_MUL3);
        /* CK_PLL2 = 25/8*14 M= 43.75M*/
        rcu_pll2_config(RCU_PLL2_MUL14);
        /* turn on the oscillator PLL1 */
        rcu_osci_on(RCU_PLL2_CK);
        while(ERROR == rcu_osci_stab_wait(RCU_PLL2_CK));
        /* configure the I2S1/I2S2 clock source selection */
        rcu_i2s1_clock_config(RCU_I2S1SRC_CKPLL2_MUL2);
        rcu_i2s2_clock_config(RCU_I2S2SRC_CKPLL2_MUL2);
    }
    /* enable PLL */
    rcu_osci_on(RCU_PLL_CK);
    /* wait till PLL is ready */
    while(ERROR == rcu_osci_stab_wait(RCU_PLL_CK));
    /* select PLL as system clock source */
    rcu_system_clock_source_config(RCU_CKSYSSRC_PLL);
    /* wait till PLL is used as system clock source */
    while(RCU_SCSS_PLL != rcu_system_clock_source_get());

    /* enable I2S1 clock */
    rcu_periph_clock_enable(RCU_SPI1);
    /* enable I2S2 clock */
    rcu_periph_clock_enable(RCU_SPI2);
    /* enable GPIOA clock */
    rcu_periph_clock_enable(RCU_GPIOA);
    /* enable GPIOB clock */
    rcu_periph_clock_enable(RCU_GPIOB);
    /* enable GPIOC clock */
    rcu_periph_clock_enable(RCU_GPIOC);
    /* enable AF clock */
    rcu_periph_clock_enable(RCU_AF);
    /* enable DMA0 clock */
    rcu_periph_clock_enable(RCU_DMA0);
    /* enable DMA1 clock */
    rcu_periph_clock_enable(RCU_DMA1);
}

/*!
    \brief      configure the GPIO peripheral
    \param[in]  none
    \param[out] none
    \retval     none
*/
void gpio_config(void)
{
    /* configure I2S1 GPIO: I2S1_WS/PB12, I2S1_CK/PB13, I2S_SD/PB15 */
    gpio_init(GPIOB, GPIO_MODE_AF_PP, GPIO_OSPEED_50MHZ, GPIO_PIN_12 | GPIO_PIN_13 | GPIO_PIN_15);

    /* configure I2S2 GPIO: I2S2_WS/PA4, I2S2_CK/PC10, I2S2_SD/PC12 */
    gpio_pin_remap_config(GPIO_SPI2_REMAP, ENABLE);
    gpio_init(GPIOC, GPIO_MODE_IN_FLOATING, GPIO_OSPEED_50MHZ, GPIO_PIN_10 | GPIO_PIN_12);
    gpio_init(GPIOA, GPIO_MODE_IN_FLOATING, GPIO_OSPEED_50MHZ, GPIO_PIN_4);
}

/*!
    \brief      configure the DMA peripheral
    \param[in]  none
    \param[out] none
    \retval     none
*/
void dma_config(void)
{
    dma_parameter_struct dma_init_struct;
    dma_struct_para_init(&dma_init_struct);

    /* configure I2S1 transmit DMA: DMA0, DMA_CH4 */
    dma_deinit(DMA0, DMA_CH4);

    dma_init_struct.periph_addr  = (uint32_t)&SPI_DATA(SPI1);
    dma_init_struct.memory_addr  = (uint32_t)i2s1_send_array;
    dma_init_struct.direction    = DMA_MEMORY_TO_PERIPHERAL;
    dma_init_struct.memory_width = DMA_MEMORY_WIDTH_16BIT;
    dma_init_struct.periph_width = DMA_PERIPHERAL_WIDTH_16BIT;
    dma_init_struct.priority     = DMA_PRIORITY_LOW;
    dma_init_struct.number       = ARRAYSIZE;
    dma_init_struct.periph_inc   = DMA_PERIPH_INCREASE_DISABLE;
    dma_init_struct.memory_inc   = DMA_MEMORY_INCREASE_ENABLE;
    dma_init(DMA0, DMA_CH4, &dma_init_struct);
    /* configure DMA mode */
    dma_circulation_disable(DMA0, DMA_CH4);
    dma_memory_to_memory_disable(DMA0, DMA_CH4);

    /* configure I2S2 receive DMA: DMA1, DMA_CH0 */
    dma_deinit(DMA1, DMA_CH0);
    dma_init_struct.periph_addr  = (uint32_t)&SPI_DATA(SPI2);
    dma_init_struct.memory_addr  = (uint32_t)i2s2_receive_array;
    dma_init_struct.direction    = DMA_PERIPHERAL_TO_MEMORY;
    dma_init_struct.priority     = DMA_PRIORITY_HIGH;
    dma_init(DMA1, DMA_CH0, &dma_init_struct);
    /* configure DMA mode */
    dma_circulation_disable(DMA1, DMA_CH0);
    dma_memory_to_memory_disable(DMA1, DMA_CH0);
}

/*!
    \brief      configure the SPI peripheral
    \param[in]  none
    \param[out] none
    \retval     none
*/
void spi_config(void)
{
    /* reset I2S1/I2S2 */
    spi_i2s_deinit(SPI1);
    spi_i2s_deinit(SPI2);

    /* configure I2S1 */
    i2s_init(SPI1, I2S_MODE_MASTERTX, I2S_STD_PHILIPS, I2S_CKPL_LOW);
    i2s_psc_config(SPI1, I2S_AUDIOSAMPLE_44K, I2S_FRAMEFORMAT_DT16B_CH16B, I2S_MCKOUT_DISABLE);

    /* configure I2S2 */
    i2s_init(SPI2, I2S_MODE_SLAVERX, I2S_STD_PHILIPS, I2S_CKPL_LOW);
    i2s_psc_config(SPI2, I2S_AUDIOSAMPLE_44K, I2S_FRAMEFORMAT_DT16B_CH16B, I2S_MCKOUT_DISABLE);
}

/*!
    \brief      memory compare function
    \param[in]  src: source data pointer
    \param[in]  dst: destination data pointer
    \param[in]  length: the compare data length
    \param[out] none
    \retval     ErrStatus : ERROR or SUCCESS
*/
ErrStatus memory_compare(uint16_t *src, uint16_t *dst, uint8_t length)
{
    while(length--) {
        if(*src++ != *dst++) {
            return ERROR;
        }
    }
    return SUCCESS;
}
